import { Effect, ItemStack, Player, world } from "@minecraft/server";
import * as df from "./diamondFunctions.js"
import { getColourCodeFromName, getFormatCodeFromName } from "../vanilla/vanillaConst.js";
import { getMobIdFromName } from "../vanilla/mobList.js";
import { DEBUG } from "../config.js";

/**
 * The Modal Action Function for the Give Action.
 * @param {{selector: Player, itemId: ItemStack, amount: number}} value - The values for the give action.
 * @param {Player} player - The player that submitted.
 */
export function tdpModalGive(value, player) {
    const {selector: name, itemId, amount} = value;
    let selector;
    if (name == "self" || name == "{#SELF}") {
        selector = player;
    } else if (name == "all" || name == "{#ALL}") {
        const worldPlayers = world.getAllPlayers();
        worldPlayers.forEach(plyr => {
            selector = plyr;
            df.runGiveItem(item, player, selector);
        })
        if (DEBUG) console.log(`All players have received ${amount} of ${itemId}`)
    } else {
        selector = world.getPlayers().find(p => p.name === name);
    }
    if (!selector) return;
    const item = new ItemStack(itemId, amount);
    df.runGiveItem(item, player, selector);
    if (DEBUG) console.log(`Player ${selector.nameTag} recieved ${amount} of ${itemId.split(":").pop().replace(/^./, str => str.toUpperCase()).replace(/_/g, " ")}`);
}

/**
 * The Modal Action Function for the Add Tag Function.
 * Give a tag to the player.
 * @param {{selector: Player, tag: string}} value - The values for the add tag action.
 * @param {Player} player - The player that submitted.
 */
export function giveTagModal(value, player) {
    const {selector: name, tag} = value;
    let selector;
    if (name == "self" || name == "{#SELF}"){
        selector = player;
    } else if (name == "all" || name == "{#ALL}") {
        const worldPlayers = world.getAllPlayers();
        worldPlayers.forEach(plyr => {
            selector = plyr;
            df.runAddTag(tag, player, selector);
        })
        return;
    } else {
        selector = world.getPlayers().find(p => p.name === name);
    }
    if (!selector) return;
    df.runAddTag(tag, player, selector);
}

/**
 * The Modal Action Function for the Remove Tag Function.
 * Removes a tag from the player.
 * @param {{selector: Player, tag: string}} value - The values for the remove tag action.
 * @param {Player} player - The player that submitted.
 */
export function removeTagModal(value, player) {
    const {selector: name, tag} = value;
    let selector;
    if (name == "self" || name == "{#SELF}"){
        selector = player;
    } else if (name == "all" || name == "{#ALL}") {
        const worldPlayers = world.getAllPlayers();
        worldPlayers.forEach(plyr => {
            selector = plyr;
            df.runRemoveTag(tag, player, selector);
        })
        return;
    } else {
        selector = world.getPlayers().find(p => p.name === name);
    }
    if (!selector) return;
    df.runRemoveTag(tag, player, selector);
}

/**
 * The Modal Action Function for the Effect Action.
 * @param {{selector: Player, effects: Effect, amplifier?: number, duration?: number, particles?: boolean}} values - The values for the effect action.
 * @param {Player} player - The player that submitted.
 */
export function giveEffectModal(values, player) {
    const { selector: name, effects, amplifier, duration, particles } = values;
    let selector;
    if (name == "self" || name == "{#SELF}"){
        selector = player;
    } else if (name == "all" || name == "{#ALL}") {
        const worldPlayers = world.getAllPlayers();
        worldPlayers.forEach(plyr => {
            selector = plyr;
            df.runGiveEffect(effects, player, selector, {amplifier, duration, particles});
        })
        if (DEBUG) console.log(`Effect given to all players!\nEffect: ${effects.toLowerCase().replace(/\s+/g, "_")} as ${effects}\nAmplifier: ${amplifier}\nDuration: ${duration}\nShow Particles: ${particles}`);
        return;
    } else {
        selector = world.getPlayers().find(p => p.name === name);
    }
    if (!selector) return;
    df.runGiveEffect(effects, player, selector, {amplifier, duration, particles});
    if (DEBUG) console.log(`Effect Given!\nPlayer: ${selector.nameTag}\nEffect: ${effects.toLowerCase().replace(/\s+/g, "_")} as ${effects}\nAmplifier: ${amplifier}\nDuration: ${duration}\nShow Particles: ${particles}`);
}

/**
 * The Modal Action Function for the Scoreboard Action.
 * @param {{selector: Player, objective: string, amount: number}} values - The values for the scoreboard action.
 * @param {Player} player - The player that submitted.
 */
export function applyScoreModal(values, player) {
    const {selector: name, objective, amount} = values;
    let selector;
    if (name == "self" || name == "{#SELF}") {
        selector = player;
    } else if (name == "all" || name == "{#ALL}") {
        const worldPlayers = world.getAllPlayers();
        const obj = makeOrFindObjective(objective);
        worldPlayers.forEach(plyr => {
            selector = plyr;

            obj.addScore(selector, amount);

        })
        if (DEBUG) console.log(`All the player's score for ${JSON.stringify(objective)} was changed by ${amount}`);
        return;
    } else {
        selector = world.getPlayers().find(p => p.name === name);
    }
    if (!selector) return;
    const obj = makeOrFindObjective(objective);

    obj.addScore(selector, amount);

    const playerScore2 = obj.getScore(selector);

    if (DEBUG) console.log(`Player ${selector.nameTag}'s score for ${JSON.stringify(objective)} was changed to ${playerScore2}`);
}

/**
 * 
 * @param {{selector: Player, message: import("@minecraft/server").RawMessage, colour: colourCode}} values 
 * @param {Player} player 
 */
export function sendMessageModal(values, player) {
    const {selector: name, message, colour, format} = values;
    let colourCode = "§f";
    let formatCode = "";
    if (colour != undefined) colourCode = getColourCodeFromName(colour);
    if (format != undefined) formatCode = getFormatCodeFromName(format);
    
    let selector;
    if (name == "self" || name == "{#SELF}") {
        selector = player;
        selector.sendMessage(`${colourCode}${formatCode}${message}`)
    } else if (name == "all" || name == "{#ALL}") {
        world.sendMessage(`${colourCode}${formatCode}${message}`);
    } else {
        selector = world.getPlayers().find(p => p.name === name);
        selector.sendMessage(`${colourCode}${formatCode}${message}`)
    }
}

export function changeGMModal(values, player) {
    const {selector: name, gamemode} = values;
    let selector;
    if (name == "self" || name == "{#SELF}") {
        selector = player;
    } else if (name == "all" || name == "{#ALL}") {
        const worldPlayers = world.getAllPlayers();
        worldPlayers.forEach(plyr => {
            selector = plyr;
            df.runGamemode(gamemode, player, selector);
        });
        return;
    } else {
        selector = world.getPlayers().find(p => p.name === name);
    }
    df.runGamemode(gamemode, player, selector);
}

export function tpPlayerOrCoordsModal(values, player) {
    const {selector, destination} = values;
    if (typeof destination == "string") {
        tpPlayerToPlayerModal(values, player);
    } else if (typeof destination == "object") {
        tpPlayerToCoordModal(values, player);
    }
}

function tpPlayerToPlayerModal(values, player) {
    const {selector: name1, destination: name2} = values
    let destination = world.getPlayers().find(p => p.name === name2);
    const dimension = destination.dimension;
    let selector;
    const location = destination.location;
    if (name1 == "self" || name1 == "{#SELF}") {
        selector = player;
    } else if (name1 == "all" || name1 == "{#ALL}") {
        const worldPlayers = world.getAllPlayers();
        worldPlayers.forEach(plyr => {
            selector = plyr;
            df.runTeleportAction(location, player, selector, dimension);
        });
        if (DEBUG) console.log(`Teleporting all players to ${destination.nameTag}`)
        return;
    } else {
        selector = world.getPlayers().find(p => p.name === name1);
    }
    if (selector == destination) {
        player.sendMessage("§cError! Cannot have the Player being Teleported be the same as the Player Destination!");
        return;
    }
    df.runTeleportAction(location, player, selector, dimension)
}

function tpPlayerToCoordModal(values, player) {
    const {selector: name, destination, dimension} = values;
    let selector;
    let dim = "overworld";
    if (dimension == undefined) dim = "overworld";
    else if (dimension != undefined) dim = dimension.toLowerCase().replace(/\s+/g, "_");
    if (name == "self" || name == "{#SELF}") {
        selector = player;
    } else if (name == "all" || name == "{#ALL}") {
        const worldPlayers = world.getAllPlayers();
        worldPlayers.forEach(plyr => {
            selector = plyr;
            df.runTeleportAction(destination, player, selector, dim);
        })
        return;
    } else {
        selector = world.getPlayers().find(p => p.name === name);
    }
    df.runTeleportAction(destination, player, selector, dim);
}

export function summonMobToPlayerModal(values, player) {
    const {selector: name, entity, location} = values;
    let spawnEntity = getMobIdFromName(entity);
    if (!spawnEntity || spawnEntity == " ") {
        console.warn("No valid Entity was provided!");
        return;
    }
    let loc = location;
    let selector;
    if (name != undefined) {
        if (name == "self" || name == "{#SELF}") {
            selector = player;
        } else if (name == "all" || name == "{#ALL}") {
            const worldPlayers = world.getAllPlayers();
            worldPlayers.forEach(plyr => {
                selector = plyr;
                if (!location || location == " ") loc = selector.location;
                df.runSpawnEntity(loc, spawnEntity, player, selector);
            });
            return;
        } else {
            selector = world.getPlayers().find(p => p.name === name);
        }
        if (!location || location == " ") loc = selector.location;
        df.runSpawnEntity(loc, spawnEntity, player, selector)
    } else if (name == undefined) {
        if (!location || location == " ") {
            console.warn("No valid Location was provided!");
            return;
        }
        df.runSpawnEntity(loc, spawnEntity, player, undefined);
    }
    
}

// ------------------------------------------------
// Extra functions for local use
// ------------------------------------------------

function makeOrFindObjective(id) {
    let obj = world.scoreboard.getObjective(id);
    if (!obj) {
        const objName = id.replace(/^./, str => str.toUpperCase()).replace(/_/g, " ");
        obj =  world.scoreboard.addObjective(id, objName);
        if (DEBUG) console.info(`Objective was not found so a new objective.\nObjective id: ${id}\nObjective Name: ${objName}`);
    }
    return obj;
}